//---------------------------------------------------------------------------
// Demonstrates use of the Fpu library for interfacing with the uM-FPU V3.1
// floating point coprocessor. This example draws a simple graph of the sine
// and cosine functions to the serial monitor.
//
// @author Cam Thompson, Micromega Corporation
// <www.micromegacorp.com>
//
// @version
//  October 2, 2011
//  - updated for new SPI and FpuSerial libraries
//  December 15, 2008
//  - original version
//---------------------------------------------------------------------------

#include <SPI.h>
#include <Fpu.h>
#include <FpuSerial.h>

//-------------------- uM-FPU register definitions --------------------------

#define Angle		10				// angle
#define Point		11				// point on graph

//-------------------- setup ------------------------------------------------

void setup()
{
  Serial.begin(9600);
  Serial.println("Graph");
  
  SPI.begin();
  Fpu.begin();

  if (Fpu.sync() == SYNC_CHAR)
    FpuSerial.printVersionln();
  else
  {
    Serial.print("uM-FPU not detected");
    while(1) ; // stop if FPU not detected
  }
}

//-------------------- loop -------------------------------------------------

void loop()
{
  // graph sine from 0 to 360 degrees (2*PI radians)
  Serial.println("\r\nGraph of sine");
  Serial.println("-1        0         1");
  Serial.println("---------------------");
  for (int i = 0; i <= 20; i++)
  {
    Fpu.write(SELECTA, Angle, LOADPI, FSET0);
    Fpu.write(FMULI, 2*i, FDIVI, 20);
    Fpu.write(SIN);
    graph();
  }

  // graph cosine from 0 to 360 degrees (2*PI radians)
  Serial.println("\r\nGraph of cosine");
  Serial.println("-1        0         1");
  Serial.println("---------------------");
  for (int i = 0; i <= 20; i++)
  {
    Fpu.write(SELECTA, Angle, LOADPI, FSET0);
    Fpu.write(FMULI, 2*i, FDIVI, 20);
    Fpu.write(COS);
    graph();
  }

  Serial.println("\r\nDone.");
  while(1) ;
}

//-------------------- graph ------------------------------------------------

void graph(void) {
  char ch, p;

  // calculate point on graph for value
  Fpu.write(SELECTA, Point, FSET, Angle);
  Fpu.write(FMULI, 10, ROUND, F_FIX);
  Fpu.wait();
  Fpu.write(LREADBYTE);
  Fpu.readDelay();
  p = Fpu.read();

  // plot the point on the graph
  for (int i = -10; i <= 10; i++)
  {
    if (i == p) ch = '*';
    else if (i == 0) ch = '|';
    else ch = ' ';
    Serial.print(ch);
  }

  // display the angle value
  Serial.print(' ');
  Fpu.write(SELECTA, Angle);
  FpuSerial.printFloatln(85);
}
